
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_tableptr.c */


#include "fs_itype.h"


#ifdef DEBUG_TABLE_PTR

FS_VOID
dump_table_ptrs(_DS_ FILECHAR *s)
{
    TABLE_PTR_REC *rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs);

    FS_PRINTF(("STATE.server->table_ptrs: %s\n", s));

    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(rec->next))
    {
        FS_PRINTF(("lfnt=%p tag='%c%c%c%c' ptr=%p ref_count=%ld external=%d\n",
                   (LFNT *)(rec->lfnt),
                   rec->tag >> 24,
                   (rec->tag >> 16) & 0xFF,
                   (rec->tag >>  8) & 0xFF,
                   rec->tag        & 0xFF,
                   (FS_BYTE *)(rec->ptr) *)(
                      rec->ref_count,
                      rec->external));
    }
}

#endif /* DEBUG_TABLE_PTR */

typedef struct zzz
{
    FS_BYTE *memptr;
} ZZZ;

/*****************************************************************************/

static FS_LONG
add_table_ptr(_DS_ LFNT *lfnt, FS_BOOLEAN lfntIsFntset,
              FS_BYTE *real_memptr,
              FS_ULONG tag, FS_BYTE *ptr, FS_BOOLEAN ext)
{
    TABLE_PTR_REC *rec;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TABLE_PTR_REC";
#endif
    rec = (TABLE_PTR_REC *)FSS_malloc(_PS_ sizeof(TABLE_PTR_REC));
    if (!rec)
    {
        return STATE.error;
    }
    /* fill the rec */
    rec->lfnt = lfnt;
    rec->tag = tag;
    if (ext)
    {
        rec->ptr = ptr;
    }
    else
    {
        int act3 = 0;
        if (!lfntIsFntset)
        {
            act3 = (lfnt->fontflags & FONTFLAG_ACT3) == FONTFLAG_ACT3;
        }
        if (act3)
        {
#ifdef FS_ACT3
            /* do something special with act3 ?? */
            ZZZ zzz, *ttf = &zzz;
            ttf->memptr = real_memptr;
            rec->ptr = ptr;
#endif
        }
        else /* ptr into memory where font resides */
        {
            ZZZ zzz, *ttf = &zzz;
            ttf->memptr = real_memptr;
            rec->ptr = ptr;
        }
    }
    rec->ref_count = 1;
    rec->external = ext;

    /* prepend rec to list, update the head */
    rec->next = STATE.server->table_ptrs;
    STATE.server->table_ptrs = rec;

    return SUCCESS;
}

static FS_VOID *
fntset_read(_DS_ FNTSET *fntset, FS_ULONG offset, FS_ULONG size)
{
    FS_FILE *fp;

    if (size == 0) return NULL;

    fp = FS_open(_PS_ fntset->path);
    if (fp)
    {
        FS_BYTE *p;

        /* do the file IO */
#ifdef FS_MEM_DBG
        STATE.memdbgid = "fntset_read";
#endif
        p = FSS_malloc(_PS_ size);
        if (p)
        {
            FS_ULONG r;
            FS_seek(_PS_ fp, offset, SEEK_SET);
            r = FS_read(_PS_ fp, p, size);
            if (STATE.error || (r != size))
            {
                FSS_free(_PS_ p);
                p = NULL;
            }
        }
        FS_close(_PS_ fp);

        return (FS_VOID *)p;
    }
    else
    {
        return NULL;
    }
}

/*****************************************************************************/

static FS_BYTE *
get_table_ptr(_DS_ LFNT *lfnt, FS_BOOLEAN lfntIsFntset,
              FS_BYTE *real_memptr,
              FS_ULONG tag, FS_ULONG offset, FS_ULONG size)
{
    TABLE_PTR_REC *rec;
    FS_BOOLEAN act3;
    FS_BYTE *ptr;
    FS_BOOLEAN ext;

#ifdef DEBUG_TABLE_PTR
    FS_PRINTF(("get_table_ptr('%c%c%c%c')\n",
               tag >> 24, 0xFF & (tag >> 16), 0xFF & (tag >> 8), 0xFF & tag));
#endif

    /* unless something bad happens later on */
    STATE.error = SUCCESS;

    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(rec->next))
    {
        if ((rec->lfnt == lfnt) &&
            (rec->tag == tag) &&
            (rec->ptr != 0))
        {
            /* bump ref count and return the pointer */
            rec->ref_count++;

#ifdef DEBUG_TABLE_PTR
            FS_PRINTF((" -- found in table ptr=%p\n" *)((FS_BYTE *)(rec->ptr)));
            dump_table_ptrs(_PS_ "");
#endif
            if (rec->external)
            {
                return (FS_BYTE *)(rec->ptr);
            }
            else
            {
                ZZZ zzz, *ttf = &zzz;
                ttf->memptr = real_memptr;
                return ((FS_BYTE *) rec->ptr);
            }
        }
    }
    act3 = 0;
    if (!lfntIsFntset)
    {
        if ( ((lfnt->fontflags & FONTFLAG_CCC) == FONTFLAG_CCC) ||
             ((lfnt->fontflags & FONTFLAG_DDD) == FONTFLAG_DDD) )
        {
            /* some CCC tables wouldn't make sense to the User */
            if ((tag == TAG_loca) || (tag == TAG_hmtx) || (tag == TAG_glyf))
            {
                STATE.error = ERR_CCC_BAD_TABLE;
                return 0;
            }
        }
        act3 = (FS_BOOLEAN)((lfnt->fontflags & FONTFLAG_ACT3) == FONTFLAG_ACT3);
    }
    if (real_memptr)
    {
        if (act3)
        {
#ifdef FS_ACT3
            /* !!! this assumes the current implementation of compression !!! */
            MTX_RA_TT_Decomp *decomp = (MTX_RA_TT_Decomp *)(FS_VOID *)(((TTF *)(FS_VOID *)(lfnt->fnt))->decomp);

            ptr = MTX_RA_Get_TTF_FragmentPtr(_PS_ decomp, offset, &size);
            ext = (FS_BOOLEAN)((ptr < decomp->cmpStart) || (ptr > decomp->cmpEnd));
#else
            /* there is an ACT3 file but FS_ACT3 undefined ??? */
            STATE.error = ERR_ACT3_UNDEF;
            return 0;
#endif /* FS_ACT3 */
        }
        else /* uncompressed ... can simply point to the table in memory */
        {
            FS_ULONG dOffset;
            dOffset = lfntIsFntset ? 0 : lfnt->data_offset;
            ptr = real_memptr + dOffset + offset;
            ext = 0;
        }
    }
    else /* file based ... it's gonna be external */
        /* even when we later implement disk based act3 files */
    {
        /* read the table */
        if (lfntIsFntset)
        {
            ptr = fntset_read(_PS_ (FNTSET *)lfnt, offset, size);
        }
        else
        {
#ifdef FS_MEM_DBG
            char tagstr[5];
            tagstr[0] = (char)(       (tag >> 24));
            tagstr[1] = (char)(0xFF & (tag >> 16));
            tagstr[2] = (char)(0xFF & (tag >>  8));
            tagstr[3] = (char)(0xFF & (tag      ));
            tagstr[4] = 0;
            STATE.memdbgid = tagstr;
#endif
            ptr = ttf_read(_PS_ (TTF *)(lfnt->fnt), offset, size);
        }
        ext = 1;
    }
    if (ptr) /* if ptr is null then STATE.error will be set */
    {
        add_table_ptr(_PS_ lfnt, lfntIsFntset, real_memptr, tag, ptr, ext);
    }
    if (STATE.error)
    {
        /* we got the pointer but were not able to create a */
        /* entry in the list, so there would be no way to free the new */
        /* pointer ... so we have to fail.  This seems unsatisfactory */
        /* but I see no way around it... SWP */
        if (ext)
        {
            FSS_free(_PS_ ptr);
        }
        ptr = 0;
    }
#ifdef DEBUG_TABLE_PTR
    else
    {
        dump_table_ptrs(_PS_ "added a new pointer");
    }
#endif

    return ptr;
}

static LFNT *rec_get_source_lfnt(TABLE_PTR_REC *rec,
                                 FNTSET *fntset, FS_BOOLEAN *lfntIsFntset)
{
    LFNT *lfnt;
    {
        lfnt = (LFNT *)(fntset->metric_font);
        *lfntIsFntset = 0;

        switch (rec->tag)
        {
        case TAG_maxp:
            if (fntset->maxp_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_head:
            if (fntset->head_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_hhea:
            if (fntset->hhea_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_OS2:
            if (fntset->os_2_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_post:
            if (fntset->post_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_vhea:
            if (fntset->vhea_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_name:
            if (fntset->name_offset)
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_cmap:
            if ((FNTSET_OT_TABLE *)(fntset->cmap))
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            break;
        case TAG_GDEF:
            if ((FNTSET_OT_TABLE *)(fntset->gdef))
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            else if (fntset->num_fonts > 0)
            {
                lfnt = fntset->cfnt[0].lfnt; /* the first component */
            }
            break;
        case TAG_GSUB:
            if ((FNTSET_OT_TABLE *)(fntset->gsub))
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            else if (fntset->num_fonts > 0)
            {
                lfnt = fntset->cfnt[0].lfnt; /* the first component */
            }
            break;
        case TAG_GPOS:
            if ((FNTSET_OT_TABLE *)(fntset->gpos))
            {
                lfnt = (LFNT *)fntset;
                *lfntIsFntset = 1;
            }
            else if (fntset->num_fonts > 0)
            {
                lfnt = fntset->cfnt[0].lfnt; /* the first component */
            }
            break;
        default:
            break;
        }
    }
    return lfnt;
}

/****************************************************************/
/************** exports *****************************************/
/****************************************************************/

FS_BYTE *FSS_get_table(_DS_ FS_ULONG tag, FS_BYTE mode, FS_ULONG *len)
{
    FNTSET *fntset;
    LFNT *lfnt;
    FS_BOOLEAN lfntIsFntset;
    FS_BYTE *real_memptr;
    FS_ULONG offset = 0;

    if ((mode != TBL_QUERY) && (mode != TBL_EXTRACT))
    {
        return 0;
    }
    fntset = STATE.cur_typeset.fntset;
    if (!fntset)
    {
        STATE.error = ERR_NO_CURRENT_LFNT;
        return 0;
    }
    {
        FNTSET_OT_TABLE *ot;

        lfnt = (LFNT *)(fntset->metric_font);

        ot = NULL;

        switch (tag)
        {
        case TAG_maxp:
        case TAG_head:
        case TAG_hhea:
        case TAG_OS2:
        case TAG_vhea:
        case TAG_post:
        case TAG_name:
            break;
        case TAG_cmap:
            ot = (FNTSET_OT_TABLE *)(fntset->cmap);
            break;
        case TAG_GDEF:
            ot = (FNTSET_OT_TABLE *)(fntset->gdef);
            break;
        case TAG_GSUB:
            ot = (FNTSET_OT_TABLE *)(fntset->gsub);
            break;
        case TAG_GPOS:
            ot = (FNTSET_OT_TABLE *)(fntset->gpos);
            break;
        default:
            break;
        }
        lfntIsFntset = 0;

        switch (tag)
        {
        case TAG_maxp:
            if (fntset->maxp_offset)
            {
                offset = fntset->maxp_offset;

                *len = sizeof(TTF_MAXP);

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_head:
            if (fntset->head_offset)
            {
                offset = fntset->head_offset;
                *len = sizeof(TTF_HEAD);

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_hhea:
            if (fntset->hhea_offset)
            {
                offset = fntset->hhea_offset;
                *len = sizeof(TTF_HHEA);

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_OS2:
            if (fntset->os_2_offset)
            {
                offset = fntset->os_2_offset;
                *len = sizeof(TTF_OS2);

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_post:
            if (fntset->post_offset)
            {
                offset = fntset->post_offset;
                *len = sizeof(TTF_POST);

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_vhea:
            if (fntset->vhea_offset)
            {
                offset = fntset->vhea_offset;
                *len = sizeof(TTF_VHEA);

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_name:
            if (fntset->name_offset)
            {
                offset = fntset->name_offset;
                *len = fntset->name_size;

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            break;
        case TAG_cmap:
        case TAG_GDEF:
        case TAG_GSUB:
        case TAG_GPOS:
            if (ot)
            {
                offset = ot->offset;
                *len = ot->size;

                if (mode == TBL_QUERY)
                {
                    return (FS_BYTE *)1;
                }
                else /* (mode == TBL_EXTRACT) */
                {
                    lfnt = (LFNT *)fntset;
                    lfntIsFntset = 1;
                }
            }
            else if ((tag != TAG_cmap) && (fntset->num_fonts > 0))
            {
                lfnt = fntset->cfnt[0].lfnt; /* the first component */
            }
            break;
        default:
            break;
        }
    }
    real_memptr = lfntIsFntset ? fntset->memptr : lfnt->memptr;

    if (!lfntIsFntset)
    {
        TTF *ttf;

        if (lfnt->fnt_type == PFR_TYPE)
        {
            STATE.error = ERR_TABLE_UNSUPPORTED;
            return 0;
        }
        /* ? load the TTF */
        if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
        {
            return 0;
        }
        STATE.cur_lfnt = lfnt; /* protect the current lfnt from get_some_back */
        ttf = (FS_VOID *)(lfnt->fnt);

        if (!get_ttf_table_info(_PS_ ttf, tag, &offset, len))
        {
            return 0;
        }
        if (mode == TBL_QUERY)
        {
            return (FS_BYTE *)1;
        }
    }
    /* (mode == TBL_EXTRACT) */
    {
        FS_BYTE *result;

        result = get_table_ptr(_PS_ lfnt, lfntIsFntset,
                               real_memptr,
                               tag, offset, *len);
        if (result != NULL)
        {
            STATE.ref_count++;
        }
        return result;
    }
}

/******************************************************************************
 * We don't really free it now ... just decrement the ref_count
 * the data will be free-d in <get_some_back> when we're zapping unused LFNTs
 */
FS_LONG FSS_free_table(_DS_ FS_VOID *ptr)
{
    FNTSET *fntset;
    TABLE_PTR_REC *rec;

    /* freeing a NULL pointer isn't REALLY an error */
    if (!ptr)
    {
        return SUCCESS;
    }
    fntset = STATE.cur_typeset.fntset;
    if (!fntset)
    {
        return STATE.error = ERR_NO_CURRENT_LFNT;
    }
    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(rec->next))
    {
        if (rec->ptr)
        {
            LFNT *lfnt;
            FS_BOOLEAN lfntIsFntset;

            lfnt = rec_get_source_lfnt(rec, fntset, &lfntIsFntset);

            if (rec->lfnt == lfnt)
            {
                FS_VOID *recptr;

                if (rec->external)
                {
                    recptr = (FS_BYTE *)(rec->ptr);
                }
                else
                {
                    FS_BYTE *real_memptr;

                    real_memptr = lfntIsFntset ? fntset->memptr : lfnt->memptr;

#ifdef FS_MAPPED_FONTS
                    /* If "real_memptr" is non-NULL, we assume the font is ROM-based   */
                    /* (not memory-mapped) and use "real_memptr" as the address of the */
                    /* font. Else, we assume the font is DISK-based and we must make   */
                    /* it memory-mapped.                                               */
                    if (!real_memptr)
                    {
                        if (lfntIsFntset)
                        {
                            real_memptr = MF_get_mapped_font(_PS_ fntset->path);
                        }
                        else
                        {
                            real_memptr = MF_get_mapped_font(_PS_ lfnt->path);
                        }
                        if (!real_memptr)
                        {
                            return STATE.error;
                        }
                    }
#endif
                    {
                        ZZZ zzz, *ttf = &zzz;
                        ttf->memptr = real_memptr;
                        recptr = rec->ptr;
                    }
                }
                if (recptr == ptr)
                {
                    if (rec->ref_count > 0)
                    {
                        rec->ref_count--;
#ifdef DEBUG_TABLE_PTR
                        dump_table_ptrs(_PS_ "decremented ref_count on a pointer");
#endif
                        STATE.ref_count--;

                        return STATE.error = SUCCESS;
                    }
                    else
                    {
                        /* this indicates USER error -- more free's than get's */
                        return STATE.error = ERR_FREE_BAD_COUNT;
                    }
                }
            }
        }
    }
    /* didn't find the pointer in the table */
    return STATE.error = ERR_FREE_BAD_PTR;
}

/****************************************************************/
/* removed unreferenced table pointers */
int squeeze_table_ptrs(_DS_ FNTSET *fntset)
{
    TABLE_PTR_REC *prev, *rec, *next;
    int got_some = 0;

#ifdef DEBUG_TABLE_PTR
    FS_PRINTF(("squeeze_table_ptrs(%p)\n", fntset));
    dump_table_ptrs(_PS_ "before");
#endif

    prev = NULL;
    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(next))
    {
        LFNT *lfnt;
        FS_BOOLEAN lfntIsFntset;

        next = rec->next;

        lfnt = rec_get_source_lfnt(rec, fntset, &lfntIsFntset);

        if ((rec->lfnt == lfnt) && (rec->ref_count == 0))
        {
            if (rec->external)
            {
                FSS_free(_PS_ (FS_BYTE *)(rec->ptr));
            }

            /* unlink and free */
            if ((TABLE_PTR_REC *)(STATE.server->table_ptrs) == rec)
            {
                STATE.server->table_ptrs = next;
            }
            else
            {
                if (prev)
                    prev->next = next;
            }
            FSS_free(_PS_ rec);

            got_some++;    /* maybe only the size of a TABLE_PTR_REC */
        }
        else prev = rec;
    }
#ifdef DEBUG_TABLE_PTR
    FS_PRINTF(("squeeze_table_ptrs(%p)\n", fntset));
    dump_table_ptrs(_PS_ "after");
#endif

    return got_some;
}


FS_BOOLEAN any_used_fntset_or_unshared_lfnt_table_ptrs(_DS_ FNTSET *fntset)
{
    TABLE_PTR_REC *rec;

    /* see if this FNTSET has any active entries in STATE.server->table_ptrs */
    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(rec->next))
    {
        LFNT *lfnt;
        FS_BOOLEAN lfntIsFntset;

        lfnt = rec_get_source_lfnt(rec, fntset, &lfntIsFntset);

        if ((rec->lfnt == lfnt) && (rec->ref_count != 0))
        {
            if (lfntIsFntset || (lfnt->fntset_refs < 2))
            {
                return 1;
            }
        }
    }
    return 0;
}


FS_BOOLEAN any_used_lfnt_table_ptrs(_DS_ FNTSET *fntset, LFNT *lfntOfInterest)
{
    TABLE_PTR_REC *rec;

    /* see if this FNTSET has any active entries in STATE.server->table_ptrs */
    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(rec->next))
    {
        LFNT *lfnt;
        FS_BOOLEAN lfntIsFntset;

        lfnt = rec_get_source_lfnt(rec, fntset, &lfntIsFntset);

        if ((rec->lfnt == lfnt) && (rec->ref_count != 0))
        {
            if (!lfntIsFntset && (lfnt == lfntOfInterest))
            {
                return 1;
            }
        }
    }
    return 0;
}

/******************************************************************************
 * this is ONLY to be called in FS_exit(), before killing the LFNTs
 * since it will kill all pointers regardless of their ref_count
 */
FS_VOID kill_table_ptrs(_DS0_)
{
    TABLE_PTR_REC *rec, *next;

    /* free all table pointers regardless of their ref_count */
    for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(next))
    {
        next = rec->next;

        if (rec->external)
        {
            FSS_free(_PS_ (FS_BYTE *)(rec->ptr));
        }
        FSS_free(_PS_ rec);
    }
    STATE.server->table_ptrs = 0;
}
